// ===== MODE ARTISTIQUE AVEC D3.JS =====
// Ce fichier gère la visualisation artistique avec D3.js

// Variable pour stocker l'animation des particules
let particleAnimationId = null;

function initArtisticD3() {
    console.log('Initializing artistic D3...');
    
    // Vérifier si D3 est chargé
    if (typeof d3 === 'undefined') {
        console.error('D3.js n\'est pas chargé');
        document.getElementById('artistic-d3-container').innerHTML = 
            '<div style="display:flex;justify-content:center;align-items:center;height:100%;color:#666;">D3.js non disponible</div>';
        return;
    }
    
    const container = d3.select('#artistic-d3-container');
    if (container.empty()) {
        console.error('Conteneur artistique D3 non trouvé');
        return;
    }
    
    // Nettoyer l'ancienne visualisation
    clearArtisticD3();
    
    // Dimensions
    const containerNode = container.node();
    const width = containerNode.getBoundingClientRect().width || 800;
    const height = containerNode.getBoundingClientRect().height || 500;
    
    console.log('Container size:', width, 'x', height);
    
    // Créer le SVG
    const svg = container.append('svg')
        .attr('id', 'artistic-svg')
        .attr('width', width)
        .attr('height', height);
    
    // Groupe principal avec transformation
    const g = svg.append('g')
        .attr('transform', `translate(${width / 2}, ${height / 2})`);
    
    // Créer la visualisation
    createArtisticVisualization(g, width, height);
    
    console.log('Artistic D3 initialized successfully');
}

function createArtisticVisualization(g, width, height) {
    // Récupérer les données
    const data = getArtisticData();
    
    console.log('Artistic data:', data);
    
    if (!data || data.length === 0) {
        console.warn('Pas de données, utilisation des données par défaut');
        // Données par défaut
        data = [
            { label: 'TF1', value: 45, type: 'tv' },
            { label: 'France 2', value: 48, type: 'tv' },
            { label: 'France 3', value: 42, type: 'tv' },
            { label: 'ARTE', value: 52, type: 'tv' },
            { label: 'M6', value: 46, type: 'tv' },
            { label: 'Canal+', value: 50, type: 'tv' }
        ];
    }
    
    // Échelles
    const maxValue = d3.max(data, d => d.value) || 100;
    const radiusScale = d3.scaleLinear()
        .domain([0, maxValue])
        .range([30, Math.min(width, height) / 3 - 20]);
    
    const colorScale = d3.scaleSequential()
        .domain([0, data.length - 1])
        .interpolator(d3.interpolateRainbow);
    
    // Fond sombre pour contraste
    g.append('rect')
        .attr('x', -width / 2)
        .attr('y', -height / 2)
        .attr('width', width)
        .attr('height', height)
        .attr('fill', 'rgba(0, 0, 0, 0.1)');
    
    // Créer des cercles concentriques animés
    const circles = g.selectAll('.artistic-circle')
        .data(data)
        .enter()
        .append('circle')
        .attr('class', 'artistic-circle')
        .attr('r', 0)
        .attr('fill', 'none')
        .attr('stroke', (d, i) => colorScale(i))
        .attr('stroke-width', 4)
        .attr('opacity', 0.8);
    
    // Animation d'entrée avec easing
    circles.transition()
        .duration(1500)
        .delay((d, i) => i * 150)
        .attr('r', d => radiusScale(d.value))
        .attr('opacity', 0.6)
        .ease(d3.easeElasticOut);
    
    // Animation continue (pulsation)
    function pulse() {
        circles.transition()
            .duration(2000)
            .attr('r', d => radiusScale(d.value) * 1.05)
            .attr('opacity', 0.8)
            .transition()
            .duration(2000)
            .attr('r', d => radiusScale(d.value))
            .attr('opacity', 0.6)
            .on('end', pulse);
    }
    
    // Démarrer la pulsation après l'animation d'entrée
    setTimeout(pulse, data.length * 150 + 1500);
    
    // Ajouter des particules flottantes
    createFloatingParticles(g, width, height);
    
    // Ajouter des lignes de connexion animées
    createConnectionLines(g, data, radiusScale);
    
    // Ajouter des labels
    addArtisticLabels(g, data, radiusScale, width, height);
}

function createFloatingParticles(g, width, height) {
    const particleCount = 30;
    const particles = [];
    
    for (let i = 0; i < particleCount; i++) {
        particles.push({
            x: (Math.random() - 0.5) * width * 0.8,
            y: (Math.random() - 0.5) * height * 0.8,
            vx: (Math.random() - 0.5) * 1.5,
            vy: (Math.random() - 0.5) * 1.5,
            radius: Math.random() * 2 + 1
        });
    }
    
    const particleGroup = g.append('g').attr('class', 'particles');
    
    const particleElements = particleGroup.selectAll('.particle')
        .data(particles)
        .enter()
        .append('circle')
        .attr('class', 'particle')
        .attr('r', d => d.radius)
        .attr('fill', '#fff')
        .attr('opacity', 0.4);
    
    function animateParticles() {
        particles.forEach(p => {
            p.x += p.vx;
            p.y += p.vy;
            
            const boundaryX = width / 2 - 20;
            const boundaryY = height / 2 - 20;
            
            if (Math.abs(p.x) > boundaryX) p.vx *= -1;
            if (Math.abs(p.y) > boundaryY) p.vy *= -1;
        });
        
        particleElements
            .attr('cx', d => d.x)
            .attr('cy', d => d.y);
        
        particleAnimationId = requestAnimationFrame(animateParticles);
    }
    
    animateParticles();
}

function createConnectionLines(g, data, radiusScale) {
    if (data.length < 2) return;
    
    const lineGroup = g.append('g').attr('class', 'connection-lines');
    
    // Trier les données par valeur
    const sortedData = [...data].sort((a, b) => a.value - b.value);
    
    // Créer des lignes entre les cercles
    for (let i = 0; i < sortedData.length - 1; i++) {
        const angle1 = (i / sortedData.length) * Math.PI * 2;
        const angle2 = ((i + 1) / sortedData.length) * Math.PI * 2;
        const r1 = radiusScale(sortedData[i].value);
        const r2 = radiusScale(sortedData[i + 1].value);
        
        lineGroup.append('line')
            .attr('class', 'connection-line')
            .attr('x1', Math.cos(angle1) * r1)
            .attr('y1', Math.sin(angle1) * r1)
            .attr('x2', Math.cos(angle2) * r2)
            .attr('y2', Math.sin(angle2) * r2)
            .attr('stroke', '#6bccb2')
            .attr('stroke-width', 1.5)
            .attr('opacity', 0)
            .transition()
            .duration(1000)
            .delay(i * 100)
            .attr('opacity', 0.4);
    }
}

function addArtisticLabels(g, data, radiusScale, width, height) {
    const labelGroup = g.append('g').attr('class', 'artistic-labels');
    
    data.forEach((d, i) => {
        const angle = (i / data.length) * Math.PI * 2 - Math.PI / 2;
        const r = radiusScale(d.value) + 35;
        const x = Math.cos(angle) * r;
        const y = Math.sin(angle) * r;
        
        // Nom de la chaîne
        labelGroup.append('text')
            .attr('x', x)
            .attr('y', y - 8)
            .attr('text-anchor', 'middle')
            .attr('fill', '#333')
            .attr('font-size', '11px')
            .attr('font-weight', '600')
            .attr('opacity', 0)
            .text(d.label)
            .transition()
            .duration(800)
            .delay(i * 100 + 500)
            .attr('opacity', 1);
        
        // Pourcentage
        labelGroup.append('text')
            .attr('x', x)
            .attr('y', y + 10)
            .attr('text-anchor', 'middle')
            .attr('fill', '#ff6b8b')
            .attr('font-size', '14px')
            .attr('font-weight', 'bold')
            .attr('opacity', 0)
            .text(d.value + '%')
            .transition()
            .duration(800)
            .delay(i * 100 + 700)
            .attr('opacity', 1);
    });
}

function getArtisticData() {
    // Récupérer les données depuis CSV_DATA
    if (!window.CSV_DATA) {
        console.warn('CSV_DATA non disponible');
        return [];
    }
    
    const yearData = window.CSV_DATA?.years;
    if (!yearData) return [];
    
    const year = document.getElementById('yearFilter')?.value || '2019';
    const tvData = yearData.tv?.[year];
    const radioData = yearData.radio?.[year];
    
    const data = [];
    
    // Ajouter les données TV (jusqu'à 6 chaînes)
    if (tvData) {
        let count = 0;
        Object.entries(tvData).forEach(([channel, value]) => {
            if (count < 6 && typeof value === 'number') {
                data.push({
                    label: channel.length > 8 ? channel.substring(0, 8) + '.' : channel,
                    value: Math.round(value),
                    type: 'tv'
                });
                count++;
            }
        });
    }
    
    // Si pas assez de données TV, ajouter des données par défaut
    if (data.length < 3) {
        const defaults = [
            { label: 'TF1', value: 45, type: 'tv' },
            { label: 'F2', value: 48, type: 'tv' },
            { label: 'F3', value: 42, type: 'tv' },
            { label: 'ARTE', value: 52, type: 'tv' },
            { label: 'M6', value: 46, type: 'tv' },
            { label: 'C+', value: 50, type: 'tv' }
        ];
        
        defaults.forEach(d => {
            if (!data.some(existing => existing.label === d.label)) {
                data.push(d);
            }
        });
    }
    
    console.log('Artistic data retrieved:', data);
    return data;
}

// Fonction pour nettoyer la visualisation
function clearArtisticD3() {
    // Arrêter l'animation des particules
    if (particleAnimationId) {
        cancelAnimationFrame(particleAnimationId);
        particleAnimationId = null;
    }
    
    // Supprimer le SVG
    d3.select('#artistic-svg').remove();
}

// Fonction pour rafraîchir la visualisation
function refreshArtisticD3() {
    console.log('Refreshing artistic D3...');
    clearArtisticD3();
    initArtisticD3();
}

// Export des fonctions
window.initArtisticD3 = initArtisticD3;
window.refreshArtisticD3 = refreshArtisticD3;
window.clearArtisticD3 = clearArtisticD3;
window.getArtisticData = getArtisticData;
